/*
 * Decompiled with CFR 0.152.
 */
package com.boom.netty.service.file;

import com.boom.netty.common.MessageSender;
import com.boom.netty.digest.GeneralDigest;
import com.boom.netty.service.file.FileDigestCached;
import com.boom.netty.service.file.FileTransfer;
import com.boom.netty.service.file.FileTransferState;
import com.boom.netty.service.file.FileUtils;
import com.boom.netty.service.file.OperationCompletion;
import com.boom.netty.service.file.msg.FileChunkMsg;
import com.boom.netty.service.file.msg.FileFooterMsg;
import com.boom.netty.service.file.msg.FileHeaderMsg;
import com.boom.netty.service.file.msg.FileTransferAckMsg;
import com.boom.netty.service.file.msg.FileTransferFailedMsg;
import com.boom.netty.service.file.msg.GetNextFileChunkMsg;
import com.boom.netty.service.file.msg.SendFileMsg;
import com.boom.netty.ws.mhf.annotation.MessageHandlerMethod;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSendHandler {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileSendHandler.class);
    private static final String COULD_NOT_READ_FILE = "Could not read file";
    private Map<String, FileTransfer> operationIdToFileTransferMap = new ConcurrentHashMap<String, FileTransfer>();
    private Map<String, Consumer<OperationCompletion>> operationIdToCompletionConsumerMap = new ConcurrentHashMap<String, Consumer<OperationCompletion>>();
    private ScheduledExecutorService timeoutCheckScheduler = null;
    private long ftTimeoutMillis;
    private long timeoutCheckCycleMillis;
    private MessageSender messageSender;
    private String baseDir;
    private Map<String, FileDigestCached> cache = new ConcurrentHashMap<String, FileDigestCached>();

    public FileSendHandler(MessageSender messageSender, String baseDir, long timeoutMillis, long timeoutCheckCycleMillis) {
        this.messageSender = messageSender;
        this.baseDir = baseDir;
        this.ftTimeoutMillis = timeoutMillis;
        this.timeoutCheckCycleMillis = timeoutCheckCycleMillis;
        this.startTimeoutChecks();
    }

    public void uploadFile(String deviceId, String fileName, String uploadDirectory, String operationId, int priority, long offset, int bytesOverlap, Consumer<OperationCompletion> completionConsumer) {
        this.uploadFile(deviceId, fileName, uploadDirectory, operationId, priority, offset, bytesOverlap, FileTransfer.DEFAULT_CHUNK_SIZE, completionConsumer);
    }

    public void uploadFile(String deviceId, String fileName, String uploadDirectory, String operationId, int priority, long offset, int bytesOverlap, int chunkSize, Consumer<OperationCompletion> completionConsumer) {
        boolean isFileOk;
        Path sourceFilePath = Paths.get(this.baseDir, fileName).toAbsolutePath();
        if (this.operationIdToFileTransferMap.containsKey(operationId)) {
            LOGGER.debug("Duplicate upload trigger for operation {}", (Object)operationId);
            if (completionConsumer != null) {
                completionConsumer.accept(new OperationCompletion(deviceId, operationId, false, 270, "Another transfer with same id is running for file: " + operationId));
            }
            return;
        }
        Set<String> foundOperations = this.getAlreadyRunning(deviceId, sourceFilePath, uploadDirectory);
        if (!foundOperations.isEmpty()) {
            if (completionConsumer != null) {
                completionConsumer.accept(new OperationCompletion(deviceId, operationId, false, 268, "Another transfer is running for file " + foundOperations.stream().findFirst()));
            }
            return;
        }
        if (this.operationIdToCompletionConsumerMap != null && completionConsumer != null) {
            this.operationIdToCompletionConsumerMap.put(operationId, completionConsumer);
        }
        if (isFileOk = this.prepareFileForSend(deviceId, sourceFilePath.toFile(), uploadDirectory, operationId, priority, offset, bytesOverlap, chunkSize, completionConsumer)) {
            this.sendNextFTMessage(operationId, deviceId);
        }
    }

    public Set<String> getAlreadyRunning(String targetId, Path sourceFilePath, String targetDir) {
        try {
            return this.operationIdToFileTransferMap.values().stream().filter(fileTransfer -> fileTransfer.getDeviceId().equals(targetId)).filter(fileTransfer -> fileTransfer.getFile().toPath().equals(sourceFilePath)).filter(fileTransfer -> fileTransfer.getDirectory().equals(targetDir)).map(FileTransfer::getOperationId).collect(Collectors.toSet());
        }
        catch (Exception e) {
            LOGGER.debug(" Unexpected error in GAR", e);
            return Collections.emptySet();
        }
    }

    public void cancelFileTransfer(String operationId) {
        FileTransfer fileTransfer = this.operationIdToFileTransferMap.get(operationId);
        if (fileTransfer != null) {
            LOGGER.info("File transfer has been canceled op: {}", (Object)operationId);
            this.dropFileTransfer(fileTransfer, false, 266, "Canceled locally");
            FileTransferFailedMsg ackMsg = new FileTransferFailedMsg(fileTransfer.getDeviceId(), operationId, 266, "Canceled");
            this.messageSender.sendMessage(ackMsg);
        }
    }

    @MessageHandlerMethod
    public void handle(FileTransferFailedMsg msg) {
        FileTransfer fileTransfer;
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("{}", (Object)msg);
        }
        if ((fileTransfer = this.operationIdToFileTransferMap.get(msg.getOperationId())) != null) {
            LOGGER.info("File transfer has been canceled op: {}", (Object)msg.getOperationId());
            this.dropFileTransfer(fileTransfer, false, msg.getCode(), msg.getReason());
        }
    }

    @MessageHandlerMethod
    public void handleGetFile(SendFileMsg msg) {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("{}", (Object)msg);
        }
        try {
            Path path = Paths.get(this.baseDir, msg.getFileName());
            if (Files.notExists(path, new LinkOption[0])) {
                throw new FileNotFoundException(path.toString());
            }
            Set<String> alreadyRunning = this.getAlreadyRunning(msg.getSource(), path.toAbsolutePath(), msg.getDownloadDir());
            if (!alreadyRunning.contains(msg.getOperationId())) {
                this.prepareFileForSend(msg.getSource(), path.toFile(), msg.getDownloadDir(), msg.getOperationId(), msg.getPriority(), msg.getOffset(), msg.getBytesOverlap(), msg.getChunkSize(), null);
                this.sendNextFTMessage(msg.getOperationId(), msg.getSource());
            } else if (alreadyRunning.size() == 1 && alreadyRunning.contains(msg.getOperationId())) {
                this.sendNextFTMessage(msg.getOperationId(), msg.getSource());
            } else {
                LOGGER.debug("Operation {} bumps into conflicted operations: {}", (Object)msg, (Object)alreadyRunning.stream().map(id -> this.operationIdToFileTransferMap.get(id)).collect(Collectors.toList()));
                alreadyRunning.forEach(id -> {
                    if (msg.getOperationId().equals(id)) {
                        this.sendNextFTMessage(msg.getOperationId(), msg.getSource());
                    } else {
                        this.cancelFileTransfer((String)id);
                    }
                });
            }
        }
        catch (Exception e) {
            LOGGER.debug("Can't process SendFileMsg. {}", (Object)msg, (Object)e);
            FileTransfer ft = this.operationIdToFileTransferMap.get(msg.getOperationId());
            if (ft != null) {
                this.dropFileTransfer(ft, false, 269, "Path " + msg.getFileName() + " not found or invalid.");
            }
            FileTransferFailedMsg cancelMsg = new FileTransferFailedMsg(msg.getSource(), msg.getOperationId(), 261, "Could not read file " + e.getMessage());
            this.messageSender.sendMessage(cancelMsg);
        }
    }

    public boolean prepareFileForSend(String targetDeviceId, File file, String directory, String operationId, int priority, long offset, int bytesOverlap, int chunkSize, Consumer<OperationCompletion> completionConsumer) {
        boolean success;
        block7: {
            success = false;
            FileTransfer transfer = null;
            try {
                Path fPath = file.toPath();
                if (Files.notExists(fPath, new LinkOption[0])) {
                    this.cache.remove(fPath.toString());
                    throw new FileNotFoundException(fPath.toString());
                }
                FileDigestCached cached = this.cache.get(fPath.toString());
                GeneralDigest openDigest = null;
                if (cached != null) {
                    openDigest = cached.getSHA256(fPath);
                }
                if (openDigest == null) {
                    openDigest = FileUtils.getDigestOfFile(fPath, "SHA-256");
                    this.cache.put(fPath.toString(), new FileDigestCached(openDigest, file.length(), file.lastModified()));
                }
                transfer = new FileTransfer(file, targetDeviceId, operationId, true, bytesOverlap, offset);
                transfer.setPriority(priority);
                GeneralDigest copiedDigest = openDigest.copy();
                transfer.setDigest(copiedDigest);
                transfer.setFileHash(copiedDigest.asString());
                transfer.setDirectory(directory);
                transfer.setChunkSize(chunkSize);
                this.operationIdToFileTransferMap.put(operationId, transfer);
                success = true;
            }
            catch (FileNotFoundException | NoSuchFileException e) {
                LOGGER.warn(COULD_NOT_READ_FILE, e);
                this.notifyCompletionConsumer(targetDeviceId, operationId, false, 269, "File " + file.getName() + " not found.");
            }
            catch (Exception e) {
                LOGGER.warn(COULD_NOT_READ_FILE, e);
                if (transfer == null) break block7;
                this.dropFileTransfer(transfer, false, 261, "Could not read file. " + e.getMessage());
            }
        }
        if (!success) {
            FileTransferFailedMsg cancelMsg = new FileTransferFailedMsg(targetDeviceId, operationId, 261, COULD_NOT_READ_FILE);
            this.messageSender.sendMessage(cancelMsg);
        }
        return success;
    }

    public void sendNextFTMessage(String operationId, String deviceId) {
        FileTransfer transfer = this.operationIdToFileTransferMap.get(operationId);
        if (transfer != null) {
            switch (transfer.getState()) {
                case SEND_HEADER: {
                    this.sendHeader(transfer);
                    break;
                }
                case SEND_CHUNK: {
                    this.sendChunk(transfer);
                    break;
                }
                case SEND_FOOTER: {
                    this.sendFooter(transfer);
                }
            }
        } else {
            FileTransferFailedMsg cancelMsg = new FileTransferFailedMsg(deviceId, operationId, 266, "Canceled by originator");
            this.messageSender.sendMessage(cancelMsg);
        }
    }

    public void dropFileTransfer(FileTransfer transfer, boolean successful, int code, String description) {
        String operationId = transfer.getOperationId();
        LOGGER.debug("Dropping FileTransfer op: {}", (Object)operationId);
        this.operationIdToFileTransferMap.remove(operationId);
        try {
            transfer.close();
        }
        catch (IOException e) {
            LOGGER.debug("Could not close file deviceId {} op: {}, file {}", transfer.getDeviceId(), operationId, transfer.getFile().getAbsolutePath());
        }
        this.notifyCompletionConsumer(transfer.getDeviceId(), operationId, successful, code, description);
    }

    public void dropAllFileTransfers() {
        ArrayList<String> operationIds = new ArrayList<String>(this.operationIdToFileTransferMap.keySet());
        for (String operationId : operationIds) {
            FileTransfer fileTransfer = this.operationIdToFileTransferMap.get(operationId);
            if (fileTransfer == null) continue;
            this.dropFileTransfer(fileTransfer, false, 266, "Cancel all");
        }
    }

    private void notifyCompletionConsumer(String deviceId, String operationId, boolean successful, int code, String description) {
        Consumer<OperationCompletion> operationCompletionConsumer = this.operationIdToCompletionConsumerMap.get(operationId);
        if (operationCompletionConsumer != null) {
            operationCompletionConsumer.accept(new OperationCompletion(deviceId, operationId, successful, code, description));
        }
        this.operationIdToCompletionConsumerMap.remove(operationId);
    }

    private void sendHeader(FileTransfer transfer) {
        FileHeaderMsg msg = new FileHeaderMsg(transfer.getDeviceId(), transfer.getOperationId(), transfer.getFile().getName(), transfer.getDirectory(), transfer.getFileSize(), transfer.getFileHash(), transfer.getDigest().getAlgorithmName());
        msg.setPriority(transfer.getPriority());
        msg.setOffset(transfer.getOffset());
        msg.setBytesOverlap(transfer.getBytesOverlap());
        LOGGER.info("Sending file header for {} op: {} to agt: {}", transfer.getFile(), transfer.getOperationId(), transfer.getDeviceId());
        this.messageSender.sendMessage(msg);
        transfer.setState(FileTransferState.SEND_CHUNK);
    }

    private void sendChunk(FileTransfer transfer) {
        long offset = transfer.getOffset();
        String operationId = transfer.getOperationId();
        if (offset < transfer.getFileSize()) {
            FileChunkMsg msg = new FileChunkMsg(transfer.getDeviceId(), operationId, offset);
            LOGGER.trace("Sending file chunk for {} op: {}, offset: {} to agt: {}", transfer.getFile(), transfer.getOperationId(), offset, transfer.getDeviceId());
            byte[] buffer = new byte[transfer.getChunkSize()];
            try {
                int readBytes = transfer.read(buffer, transfer.getChunkSize());
                if (readBytes == transfer.getChunkSize()) {
                    msg.setBytesPayload(buffer);
                } else {
                    msg.setBytesPayload(Arrays.copyOfRange(buffer, 0, readBytes));
                }
                if (transfer.getOffset() == transfer.getFileSize()) {
                    transfer.setState(FileTransferState.SEND_FOOTER);
                }
                this.messageSender.sendMessage(msg);
            }
            catch (Exception e) {
                LOGGER.warn("Could not send message chunk. op: {}", (Object)operationId);
                FileTransferFailedMsg cancelMsg = new FileTransferFailedMsg(transfer.getDeviceId(), operationId, 261, COULD_NOT_READ_FILE);
                this.messageSender.sendMessage(cancelMsg);
                this.dropFileTransfer(transfer, false, cancelMsg.getCode(), cancelMsg.getReason());
            }
        } else {
            this.sendFooter(transfer);
        }
    }

    private void sendFooter(FileTransfer transfer) {
        FileFooterMsg msg = new FileFooterMsg(transfer.getDeviceId(), transfer.getOperationId());
        try {
            transfer.close();
        }
        catch (Exception e) {
            LOGGER.debug("Could not close file {}. op: {}", transfer.getFile() == null ? "null" : transfer.getFile().toPath(), transfer.getOperationId(), e);
        }
        LOGGER.trace("Sending file footer for {} op: {} to agt: {}", transfer.getFile(), transfer.getOperationId(), transfer.getDeviceId());
        this.messageSender.sendMessage(msg);
        if (this.cache.size() > 10000) {
            this.cache.clear();
        } else {
            this.cache.entrySet().removeIf(en -> ((FileDigestCached)en.getValue()).createdMs + TimeUnit.DAYS.toMillis(2L) < System.currentTimeMillis());
        }
    }

    @MessageHandlerMethod
    public void handle(FileTransferAckMsg msg) {
        String operationId = msg.getOperationId();
        if (msg.isSuccessful()) {
            LOGGER.info("File successfully transferred op: {}", (Object)operationId);
        } else {
            LOGGER.warn("File transfer failed. op: {}", (Object)operationId);
        }
        FileTransfer fileTransfer = this.operationIdToFileTransferMap.get(operationId);
        if (fileTransfer != null) {
            this.dropFileTransfer(fileTransfer, msg.isSuccessful(), msg.getCode(), msg.getErrorDescription());
        }
    }

    @MessageHandlerMethod
    public void handle(GetNextFileChunkMsg msg) {
        LOGGER.trace("sending file chunk. offset={}, op:{}", (Object)msg.getOffset(), (Object)msg.getOperationId());
        FileTransfer transfer = this.operationIdToFileTransferMap.get(msg.getOperationId());
        if (transfer != null) {
            transfer.setPriority(msg.getPriority());
            transfer.setChunkSize(msg.getChunkSize());
            if (msg.getOffset() != transfer.getOffset()) {
                transfer.setOffset(msg.getOffset());
                try {
                    transfer.initialize();
                }
                catch (Exception e) {
                    LOGGER.warn("Could not initialize reader", e);
                    FileTransferFailedMsg cancelMsg = new FileTransferFailedMsg(transfer.getDeviceId(), msg.getOperationId(), 261, COULD_NOT_READ_FILE);
                    this.messageSender.sendMessage(cancelMsg);
                    this.dropFileTransfer(transfer, false, cancelMsg.getCode(), cancelMsg.getReason());
                    return;
                }
            }
        }
        this.sendNextFTMessage(msg.getOperationId(), msg.getSource());
    }

    private void checkTimeouts() {
        long currentMillis = System.currentTimeMillis();
        List timedOutOperationIds = this.operationIdToFileTransferMap.entrySet().stream().filter(e -> ((FileTransfer)e.getValue()).getState() != FileTransferState.TIME_OUT && ((FileTransfer)e.getValue()).getLastUpdateTimestamp() + this.ftTimeoutMillis < currentMillis).map(Map.Entry::getKey).collect(Collectors.toList());
        for (String timedOutOperationId : timedOutOperationIds) {
            FileTransfer fileTransfer = this.operationIdToFileTransferMap.get(timedOutOperationId);
            if (fileTransfer == null) continue;
            fileTransfer.setState(FileTransferState.TIME_OUT);
            this.messageSender.executeInDeviceWorkerThread(fileTransfer.getDeviceId(), () -> {
                LOGGER.warn("Operation timed out. op: {}", (Object)timedOutOperationId);
                this.dropFileTransfer(fileTransfer, false, 265, "Operation timed out. file=" + fileTransfer.getFile() == null ? "null?" : fileTransfer.getFile().getName() + " lastupdate=" + fileTransfer.getLastUpdateTimestamp() + " size=" + fileTransfer.getFileSize() + " offset=" + fileTransfer.getOffset() + " hash=" + fileTransfer.getFileHash());
            });
        }
    }

    public void stopTimeoutChecks() {
        if (this.timeoutCheckScheduler != null && !this.timeoutCheckScheduler.isShutdown()) {
            this.timeoutCheckScheduler.shutdown();
        }
    }

    public void startTimeoutChecks() {
        if (this.timeoutCheckScheduler == null || this.timeoutCheckScheduler.isShutdown()) {
            this.timeoutCheckScheduler = Executors.newScheduledThreadPool(1);
            this.timeoutCheckScheduler.scheduleAtFixedRate(this::checkTimeouts, this.timeoutCheckCycleMillis, this.timeoutCheckCycleMillis, TimeUnit.MILLISECONDS);
        }
    }

    public void clearOperationsByDeviceId(String deviceId) {
        List<FileTransfer> operations = this.getOperationsByAgentId(deviceId);
        for (FileTransfer transfer : operations) {
            this.dropFileTransfer(transfer, false, 267, "Device " + deviceId + " disconnected");
        }
    }

    public List<FileTransfer> getOperationsByAgentId(String agentId) {
        return this.operationIdToFileTransferMap.values().stream().filter(fileTransfer -> fileTransfer.getDeviceId().equals(agentId)).collect(Collectors.toList());
    }

    public void clean() {
        HashSet<String> operationIds = new HashSet<String>(this.operationIdToFileTransferMap.keySet());
        for (String operationId : operationIds) {
            FileTransfer fileTransfer = this.operationIdToFileTransferMap.get(operationId);
            this.dropFileTransfer(fileTransfer, false, 0, "Test Cleanup");
        }
    }
}

