/*
 * Decompiled with CFR 0.152.
 */
package com.boom.netty.service.file;

import com.boom.netty.digest.FileDigestInfo;
import com.boom.netty.digest.GeneralDigest;
import com.boom.netty.service.file.FileChunkMismatchException;
import com.boom.netty.service.file.FileTransferState;
import com.boom.netty.service.file.io.FileReader;
import com.boom.netty.service.file.io.FileWriter;
import com.boom.netty.service.file.io.StreamFileReader;
import com.boom.netty.service.file.io.StreamFileWriter;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileTransfer
implements Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileTransfer.class);
    public static int DEFAULT_CHUNK_SIZE = 51200;
    private volatile FileTransferState state = FileTransferState.SEND_HEADER;
    private final String operationId;
    private int priority = 100;
    private final String deviceId;
    private final boolean readMode;
    private FileReader reader;
    private FileWriter writer;
    private long lastUpdateTimestamp;
    private int chunkSize = DEFAULT_CHUNK_SIZE;
    private boolean processing = false;
    private File file;
    private File tmpFileForWriting;
    private String directory = "";
    private long offset;
    private long fileSize;
    private String fileHash;
    private int bytesOverlap = 0;
    private byte[] overlappingBytes;
    private GeneralDigest digest;

    public FileTransfer(File file, String deviceId, String operationId, boolean readMode, int bytesOverlap, long offset) throws IOException {
        this.file = file;
        this.tmpFileForWriting = null;
        this.bytesOverlap = bytesOverlap;
        this.overlappingBytes = new byte[bytesOverlap];
        this.readMode = readMode;
        this.operationId = operationId;
        this.deviceId = deviceId;
        this.offset = offset;
        this.initialize();
    }

    public FileTransfer(File file, String deviceId, String operationId, boolean readMode, int bytesOverlap, FileDigestInfo digestInfo) throws IOException {
        this(file, deviceId, operationId, readMode, bytesOverlap, digestInfo.getOffset());
        this.setDigest(digestInfo.getDigest());
    }

    public void initialize() throws IOException {
        if (this.reader != null) {
            this.reader.close();
            LOGGER.trace("r-closed {}", (Object)this.getFile().toPath());
        }
        if (this.writer != null) {
            this.writer.close();
            LOGGER.trace("w-closed {}", (Object)this.getTmpFileForWriting().toPath());
        }
        if (this.readMode) {
            if (this.file.exists()) {
                this.fileSize = this.file.length();
                this.reader = new StreamFileReader(this.file);
                if (this.offset > (long)this.bytesOverlap && this.bytesOverlap > 0) {
                    this.reader.seek(this.offset - (long)this.bytesOverlap);
                    this.reader.read(this.overlappingBytes, this.bytesOverlap);
                } else {
                    this.reader.seek(this.offset);
                }
            } else {
                this.fileSize = 0L;
                this.reader = new StreamFileReader(this.file);
            }
            LOGGER.trace("r-opened {}", (Object)this.getFile().toPath());
        } else {
            this.readOverlappingBytesFromFile();
            this.writer = new StreamFileWriter(this.getTmpFileForWriting(), this.offset > 0L);
            LOGGER.trace("w-opened {}", (Object)this.getTmpFileForWriting().toPath());
        }
        this.lastUpdateTimestamp = System.currentTimeMillis();
    }

    private void readOverlappingBytesFromFile() throws IOException {
        File tmpFile = this.getTmpFileForWriting();
        if (this.overlappingBytes.length > 0 && tmpFile.exists() && tmpFile.length() > (long)this.overlappingBytes.length) {
            try (FileInputStream inStream = new FileInputStream(tmpFile);){
                inStream.skip(this.offset - (long)this.bytesOverlap);
                inStream.read(this.overlappingBytes);
            }
        }
    }

    public int read(byte[] buffer, int maxChunkSize) throws IOException {
        byte[] tmp = new byte[maxChunkSize];
        int readBytes = -1;
        if (this.offset > (long)this.bytesOverlap) {
            this.offset -= (long)this.bytesOverlap;
        }
        if (this.offset == 0L) {
            readBytes = this.reader.read(tmp, maxChunkSize);
            System.arraycopy(tmp, 0, buffer, 0, readBytes);
        } else {
            readBytes = this.reader.read(tmp, maxChunkSize - this.bytesOverlap);
            System.arraycopy(this.overlappingBytes, 0, buffer, 0, this.bytesOverlap);
            System.arraycopy(tmp, 0, buffer, this.bytesOverlap, readBytes);
        }
        this.overlappingBytes = readBytes > this.bytesOverlap ? Arrays.copyOfRange(tmp, readBytes - this.bytesOverlap, readBytes) : new byte[this.bytesOverlap];
        if (this.offset > (long)this.bytesOverlap) {
            readBytes += this.bytesOverlap;
        }
        this.offset += (long)readBytes;
        this.lastUpdateTimestamp = System.currentTimeMillis();
        return readBytes;
    }

    public void write(byte[] bytes) throws IOException, FileChunkMismatchException {
        if (!this.readMode) {
            if (this.overlappingPartMatches(bytes)) {
                byte[] bytesToWrite = this.offset > 0L ? Arrays.copyOfRange(bytes, this.bytesOverlap, bytes.length) : bytes;
                this.writer.write(bytesToWrite);
                this.overlappingBytes = bytes.length > this.bytesOverlap ? Arrays.copyOfRange(bytes, bytes.length - this.bytesOverlap, bytes.length) : new byte[this.bytesOverlap];
                if (this.digest != null) {
                    this.digest.update(bytesToWrite);
                }
                this.offset = this.offset == 0L ? (this.offset += (long)bytes.length) : (this.offset += (long)(bytes.length - this.bytesOverlap));
            } else {
                throw new FileChunkMismatchException("Overlapping bytes mismatch");
            }
            this.lastUpdateTimestamp = System.currentTimeMillis();
        }
    }

    private boolean overlappingPartMatches(byte[] bytes) {
        boolean chunksHaveCommonPart = true;
        if (this.bytesOverlap > 0 && this.offset > (long)this.bytesOverlap && bytes.length >= this.bytesOverlap) {
            for (int i = 0; i < this.bytesOverlap; ++i) {
                if (bytes[i] == this.overlappingBytes[i]) continue;
                chunksHaveCommonPart = false;
                break;
            }
        }
        return chunksHaveCommonPart;
    }

    @Override
    public void close() throws IOException {
        if (this.readMode && this.reader != null) {
            this.reader.close();
            LOGGER.trace("r-closed {}", (Object)this.getFile().toPath());
        } else if (this.writer != null) {
            this.writer.close();
            LOGGER.trace("w-closed {}", (Object)this.getTmpFileForWriting().toPath());
        }
    }

    public void resetOffset() {
        this.offset = 0L;
    }

    public boolean isReadMode() {
        return this.readMode;
    }

    public File getFile() {
        return this.file;
    }

    public long getOffset() {
        return this.offset;
    }

    public void setOffset(long offset) {
        this.offset = offset;
    }

    public long getFileSize() {
        return this.fileSize;
    }

    public String getFileHash() {
        return this.fileHash;
    }

    public int getBytesOverlap() {
        return this.bytesOverlap;
    }

    public byte[] getOverlappingBytes() {
        return this.overlappingBytes;
    }

    public String getOperationId() {
        return this.operationId;
    }

    public int getPriority() {
        return this.priority;
    }

    public void setPriority(int priority) {
        this.priority = priority;
    }

    public FileTransferState getState() {
        return this.state;
    }

    public String getDeviceId() {
        return this.deviceId;
    }

    public void setState(FileTransferState state) {
        this.state = state;
    }

    public void setFileSize(long fileSize) {
        this.fileSize = fileSize;
    }

    public void setFileHash(String fileHash) {
        this.fileHash = fileHash;
    }

    public GeneralDigest getDigest() {
        return this.digest;
    }

    public void setDigest(GeneralDigest digest) {
        this.digest = digest;
    }

    public void setFile(File file) {
        this.file = file;
    }

    public boolean isProcessing() {
        return this.processing;
    }

    public void setProcessing(boolean processing) {
        this.processing = processing;
    }

    public long getLastUpdateTimestamp() {
        return this.lastUpdateTimestamp;
    }

    public int getChunkSize() {
        return this.chunkSize;
    }

    public void setChunkSize(int chunkSize) {
        this.chunkSize = chunkSize;
    }

    public String getDirectory() {
        return this.directory;
    }

    public void setDirectory(String directory) {
        this.directory = directory;
    }

    public File getTmpFileForWriting() {
        if (this.tmpFileForWriting == null && this.file != null) {
            this.tmpFileForWriting = new File(this.file.getParentFile(), this.file.getName() + ".tmp");
        }
        return this.tmpFileForWriting;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        FileTransfer transfer = (FileTransfer)o;
        return this.operationId != null ? this.operationId.equals(transfer.operationId) : transfer.operationId == null;
    }

    public int hashCode() {
        return this.operationId != null ? this.operationId.hashCode() : 0;
    }

    public void cleanupTempFiles(boolean removeHashInfo) throws IOException {
        Files.deleteIfExists(this.getTmpFileForWriting().toPath());
        if (removeHashInfo) {
            Files.deleteIfExists(Paths.get(this.getFile().getAbsolutePath() + ".@@sha256", new String[0]));
        }
    }

    public void markComplete(boolean removeHashInfo) throws IOException {
        if (this.getTmpFileForWriting().exists()) {
            Files.move(this.getTmpFileForWriting().toPath(), this.getFile().toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        this.cleanupTempFiles(removeHashInfo);
    }
}

