/*
 * Decompiled with CFR 0.152.
 */
package com.blixx.server;

import com.blixx.log.RTLogger;
import com.blixx.sa.SchedulerTask;
import com.blixx.shared.Cryptor;
import com.blixx.shared.Interval;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class CSRStorage {
    private Map<String, CSRPayload> deviceIdToCSR = new ConcurrentHashMap<String, CSRPayload>();
    private final long csrTTLMs;
    private final Path csrStoreDir;
    private final Gson gson;
    private final Pattern csrNormalizationPattern = Pattern.compile("\\s+");
    private final Pattern uuidNormalizationPattern = Pattern.compile("[^abcdef0123456789-]");

    public CSRStorage(String csrStorageDir, long csrTTLMs) {
        this.csrStoreDir = Paths.get(csrStorageDir, new String[0]);
        this.gson = new GsonBuilder().setPrettyPrinting().create();
        this.csrTTLMs = csrTTLMs;
    }

    public void load() {
        if (!Files.exists(this.csrStoreDir, new LinkOption[0])) {
            try {
                Files.createDirectories(this.csrStoreDir, new FileAttribute[0]);
            }
            catch (IOException e) {
                RTLogger.print(1, "Could not create CSR storage dir {}", this.csrStoreDir.toAbsolutePath().toString(), e);
            }
        }
        try (Stream<Path> list = Files.list(this.csrStoreDir);){
            this.deviceIdToCSR = list.filter(p -> p.getFileName().endsWith(".csr")).map(this::loadCSRFromFile).filter(p -> p != null && p.getAgentId() != null).collect(Collectors.toConcurrentMap(CSRPayload::getAgentId, Function.identity()));
            this.deviceIdToCSR.values().forEach(csr -> csr.setCsr(csr.getCsr()));
            RTLogger.print(2, "CSR Storage initialized: {}", this.csrStoreDir);
        }
        catch (IOException e) {
            RTLogger.print(1, "Could not read files from directory " + this.csrStoreDir.toAbsolutePath().toString(), e);
        }
    }

    private String normalizeCsr(String csr) {
        return csr == null ? null : this.csrNormalizationPattern.matcher(csr).replaceAll("");
    }

    public boolean storeCertificateRequest(String agentID, String csrFromAgent) {
        CSRPayload valueAfterComputation;
        boolean stored = false;
        CSRPayload csr = new CSRPayload(agentID, System.currentTimeMillis(), csrFromAgent);
        if (csr == (valueAfterComputation = this.deviceIdToCSR.compute(agentID, (k, oldCSRPayload) -> oldCSRPayload == null || !this.wasRecentlySent(csrFromAgent, (CSRPayload)oldCSRPayload) ? csr : oldCSRPayload))) {
            try {
                Files.deleteIfExists(this.getCSRResponseFileForAgent(agentID));
            }
            catch (IOException e) {
                RTLogger.print(1, "Could not delete agent CSR response file agentID=" + agentID, e);
            }
            String payload = this.gson.toJson(csr);
            try {
                Files.write(this.getCSRFileForAgent(agentID), payload.getBytes(StandardCharsets.UTF_8), StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
                stored = true;
            }
            catch (IOException e) {
                RTLogger.print(1, "Could not write agent CSR to file agentID=" + agentID, e);
            }
        }
        return stored;
    }

    private boolean wasRecentlySent(String csrFromAgent, CSRPayload oldCSRPayload) {
        return oldCSRPayload.getSentToPkiTimestamp() + this.csrTTLMs > System.currentTimeMillis() && Objects.equals(this.normalizeCsr(csrFromAgent), this.normalizeCsr(oldCSRPayload.getCsr()));
    }

    private CSRPayload loadCSRFromFile(Path path) {
        CSRPayload result = null;
        try {
            byte[] bytes = Files.readAllBytes(path);
            result = this.gson.fromJson(new String(bytes), CSRPayload.class);
        }
        catch (IOException e) {
            RTLogger.print(2, "Could not read CSR from file " + path.toAbsolutePath().toString(), e);
        }
        return result;
    }

    private CSRResponsePayload loadCSRResponseFromFile(Path path) {
        CSRResponsePayload result = null;
        try {
            byte[] bytes = Files.readAllBytes(path);
            result = this.gson.fromJson(new String(bytes), CSRResponsePayload.class);
        }
        catch (IOException e) {
            RTLogger.print(2, "Could not read CSR response from file " + path.toAbsolutePath().toString(), e);
        }
        return result;
    }

    public CSRPayload removeCSRRequest(String agentId) {
        CSRPayload csrPayload = null;
        RTLogger.print(2, "Remove CSR request for agentId " + agentId);
        csrPayload = this.deviceIdToCSR.remove(agentId);
        Path path = this.getCSRFileForAgent(agentId);
        try {
            Files.deleteIfExists(path);
        }
        catch (IOException e) {
            RTLogger.print(2, "Could not remove CSR file " + path.toAbsolutePath().toString(), e);
        }
        return csrPayload;
    }

    public void removeCSRResponse(String agentId) {
        Path path2 = this.getCSRResponseFileForAgent(agentId);
        try {
            Files.deleteIfExists(path2);
        }
        catch (IOException e) {
            RTLogger.print(2, "Could not remove CSR Response file " + path2.toAbsolutePath().toString(), e);
        }
    }

    private Path getCSRFileForAgent(String agentId) {
        return this.csrStoreDir.resolve(this.normalizeFileName(agentId));
    }

    private Path getCSRResponseFileForAgent(String agentId) {
        return this.csrStoreDir.resolve(this.normalizeFileName(agentId) + ".resp");
    }

    private String normalizeFileName(String agentId) {
        return this.uuidNormalizationPattern.matcher(agentId).replaceAll("");
    }

    public Optional<String> getCSRResponse(String agentId) {
        CSRResponsePayload csrResponsePayload = this.loadCSRResponseFromFile(this.getCSRResponseFileForAgent(agentId));
        if (csrResponsePayload == null) {
            return Optional.empty();
        }
        return Optional.of(Cryptor.decrypt(csrResponsePayload.getResponse()));
    }

    public void storeCSRResponse(String agentID, byte[] signed) {
        String encryptedResponse = Cryptor.encrypt4(new String(signed));
        CSRResponsePayload resp = new CSRResponsePayload(agentID, System.currentTimeMillis(), encryptedResponse);
        String payload = this.gson.toJson(resp);
        try {
            Files.write(this.getCSRResponseFileForAgent(agentID), payload.getBytes(StandardCharsets.UTF_8), StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
        }
        catch (IOException e) {
            RTLogger.print(1, "Could not write agent CSR response to file agentID=" + agentID, e);
        }
    }

    public boolean isCsrPresent(String agentID) {
        return this.deviceIdToCSR.containsKey(agentID);
    }

    public void cleanupExpiredPKIRequests() {
        long cutOffTimeStamp = System.currentTimeMillis() - this.csrTTLMs;
        List<String> toRemove = this.deviceIdToCSR.entrySet().stream().filter(e -> ((CSRPayload)e.getValue()).getSentToPkiTimestamp() < cutOffTimeStamp).map(Map.Entry::getKey).collect(Collectors.toList());
        int numExpired = toRemove.size();
        if (numExpired > 0) {
            RTLogger.print(2, "Number of CSR requests expired " + numExpired);
        }
        toRemove.forEach(this::removeCSRRequest);
    }

    public static class CSRPayload {
        private String agentId;
        private long sentToPkiTimestamp;
        private String csr;

        public CSRPayload() {
        }

        public CSRPayload(String agentId, long sentToPkiTimestamp, String csr) {
            this.agentId = agentId;
            this.sentToPkiTimestamp = sentToPkiTimestamp;
            this.csr = csr;
        }

        public String getAgentId() {
            return this.agentId;
        }

        public void setAgentId(String agentId) {
            this.agentId = agentId;
        }

        public long getSentToPkiTimestamp() {
            return this.sentToPkiTimestamp;
        }

        public void setSentToPkiTimestamp(long sentToPkiTimestamp) {
            this.sentToPkiTimestamp = sentToPkiTimestamp;
        }

        public String getCsr() {
            return this.csr;
        }

        public void setCsr(String csr) {
            this.csr = csr;
        }
    }

    public static class CSRResponsePayload {
        private String agentId;
        private long receivedTimestamp;
        private String response;

        public CSRResponsePayload() {
        }

        public CSRResponsePayload(String agentId, long receivedTimestamp, String response) {
            this.agentId = agentId;
            this.receivedTimestamp = receivedTimestamp;
            this.response = response;
        }

        public String getAgentId() {
            return this.agentId;
        }

        public void setAgentId(String agentId) {
            this.agentId = agentId;
        }

        public long getReceivedTimestamp() {
            return this.receivedTimestamp;
        }

        public void setReceivedTimestamp(long receivedTimestamp) {
            this.receivedTimestamp = receivedTimestamp;
        }

        public String getResponse() {
            return this.response;
        }

        public void setResponse(String response) {
            this.response = response;
        }
    }

    public static class CleanupExpiredPKIRequests
    extends SchedulerTask {
        private final CSRStorage csrStorage;

        public CleanupExpiredPKIRequests(CSRStorage csrStorage, int intervalSeconds) {
            super("CleanupExpiredPKIRequests");
            this.setInterval(Interval.getSimpleInterval(intervalSeconds));
            this.csrStorage = csrStorage;
        }

        @Override
        public boolean onInit() throws Exception {
            return true;
        }

        @Override
        public boolean onStartRun() {
            return true;
        }

        @Override
        public boolean onGetData() {
            this.csrStorage.cleanupExpiredPKIRequests();
            return true;
        }

        @Override
        public boolean onStopRun() {
            return true;
        }

        @Override
        protected boolean onStopTask() {
            return true;
        }

        @Override
        public boolean refresh() {
            return true;
        }
    }
}

