#!/bin/bash
######################################################################
#
#  Name: winAgent_winServ_replace.sh
#
#  Description: Replaces Windows Service path for boom Agent (inside
#               Windows registration) by uninstall, install and start
#               service. Runaround by creating scheduled task for doing
#               this because uninstalling service aborts immediately
#               agent-server connection and prevents further remote control.
#               Scheduled task onsite Windows system executes commands
#               of file bserv.cmd so to say in background of that 
#               "lost connection situation".
#               Service uninstall and re-install is necessary for 2 reasons:
#               1. overwriting service file booma_service_64.exe (otherwise
#                  this file is locked as long as service is running)
#               2. new entry in registry of service path (surrounded with 
#                  quotes)
#
#               
#  Note: service file handling is related to Windows 64bit system
#
#  Layout bserv.cmd:
#	"TIMEOUT /T 1" 
#	"\"${agt_dir}\\${BOOMASERV}\" -u" 
#	"TIMEOUT /T 3" 
#	"copy /Y \"${agt_dir}\\510${BOOMASERV}\" \"${agt_dir}\\${BOOMASERV}\"" 
#	"TIMEOUT /T 1" 
#	"\"${agt_dir}\\${BOOMASERV}\" -i" 
#	"TIMEOUT /T 3" 
#	"\"${agt_dir}\\${BOOMASERV}\" -s" 
#
#  Note: default folder of boom Agent is sub-directory <boom_agent_instdir>\spi
#
#  Note: def. service path (BOOMAgent)
#        "C:\Program Files\boom\agent\booma_service_64.exe"
#
#
#  Author: Solmsdorf (equensWorldline)
#  Version: 1.50
#  Date: 23.2.2024
#
#  Input parameter: <windows_system>
#
#  Return: 0 = Success
#          1 = Warning (Usage ...)
#          2 = ERROR (upload boom_agt.cmd (${agt_dir}) failed!)
#          3 = ERROR (upload 510booma_service_64.exe (${agt_dir}) failed!)
#          4 = ERROR (upload bserv.cmd (${agt_dir}) failed!)
#          5 = ERROR (timeout: agent offline!)
#          6 = ERROR (get service info failed!)
#          7 = ERROR (SCHTASKS: create|run|delete task >>bServ<< failed!)
#          8 = ERROR (cmd-file >>${BSERV_CMDFIL}<< not up-to-date (modification time/file length)!)
#          9 = WARNING (node ${node_short} not available!)
#
#  History:
#
#     date    |name| reason
#  ----------------------------------------------------------------
#  23/02/2024 | ps | (1.00) Initial Release
#  29/02/2024 | ps | (1.50) ShowStar(s) if arg. 2 is empty
#
######################################################################

NAME=$(basename $0)
PID=$$

WORKDIR="/opt/boom/server/srv/cli/LinuxUtilsSamples"
USAGE="
Usage: ${NAME} <win_system_FQDN>
"

SRVDEPLOY_AGTDIR="/opt/boom/server/srv/deploy/agent"
win="win64"
BOOMASERV="booma_service_64.exe"
BOOMACMD="boom_agt.cmd"

BSERV_CMDNAM="bserv.cmd"
BSERV_CMDFIL="${WORKDIR}/${BSERV_CMDNAM}"
##MAXWAIT_CONNECT=10
MAXWAIT_CONNECT=15
WAITS=2 # sec.
WRITE_INTERV=5 # sec.
BSERV_CMDFIL_LEN=8


# FUNCTIONS++++++++++++++++++++++++++++++++++++++++++++++++++
OnDie()
{
  [[ ! -z "${ShowStarCC_PID}" ]] && {
    kill -s SIGINT ${ShowStarCC_PID} 
  }
  [[ ! -z "${ShowStarC_PID}" ]] && {
    kill -s SIGINT ${ShowStarC_PID}
  }

  ShowT WARNING "(${PID}) [${user}] ${NAME} >>${node}<< aborted!" ${LOGFILE}
  exit 0
}

function Wayout
{
  local ret=${1}

  [[ ! -z "${ShowStarCC_PID}" ]] && {
    kill -s SIGINT ${ShowStarCC_PID}
  }
  [[ ! -z "${ShowStarC_PID}" ]] && {
    kill -s SIGINT ${ShowStarC_PID}
  }


  exit ${ret}
}


function ShowT()
{
  local timstmp=$(date "+%d.%m.%Y %T")

##   echo -e "${timstmp} ${1}: ${2}"
   echo "${timstmp} ${1}: ${2}"
}


function ShowStarCC
{
  scnt=$MAXWAIT_CONNECT
(
  trap 'echo -en "\r \r";exit 0' SIGINT SIGTERM
  chars="| \\ – /"
#  delay=0.1
  delay=0.25
#  local scnt=0

  while true
  do
#    ((scnt=scnt+1))
    ((scnt=scnt-1))
    for char in $chars
    do
      sleep $delay
#      printf "\r${char}"
      printf "\r${char} (${scnt}) \r"
    done
#    sleep $delay
  done
) &

ShowStarCC_PID=$!
##echo "DBG: ShowStarCC_PID=$ShowStarCC_PID"
}

function ShowStarC
{
(
  trap 'echo -en "\r \r";exit 0' SIGINT SIGTERM
  chars="| / – \\"
  delay=0.25
  local scnt=0

  while true
  do
    ((scnt=scnt+1))
    for char in $chars
    do 
      sleep $delay
#      printf "\r${char}"
      printf "\r${char} (${scnt})\r"
    done
#    sleep $delay
  done
) &

ShowStarC_PID=$!
##echo "DBG: ShowStarC_PID=$ShowStarC_PID"
}


# INIT+++++++++++++++++++++++++++++++++++++++
trap 'OnDie' SIGINT SIGTERM
ShowStarC_PID=""
ShowStarCC_PID=""

#  Arguments?
if [ $# -lt 1 ]
then
  printf "${USAGE} \n"
  exit 1
fi
node="${1}"
ss_flag=${2}
if [ ! -z "${ss_flag}" ]
then
# arg 2 is not empty: no ShowStar
  ss_flag=0
else
  ss_flag=1
fi
node_short=$(echo ${node} | cut -f1-1 -d .)

user=$(id |awk -F"=" '{print $2}' |awk -F"(" '{print $2}' | awk -F")" '{print $1}')
cd ${WORKDIR}

chmod u+x boomCMD boomUPLD

# MAIN++++++++++++++++++++++++++++++++++++++++++

# get agent's dir: <agt_install_dir>\spi
agt_dir=$(./boomCMD ${node} "cd"); ret=$?
if [ ${ret} -ne 0 ]
then
  ShowT WARNING "(${PID}) [${user}] ${NAME} started - ${node_short} not available!"
  exit 9
fi

# truncate last "\" (..\spi)
agt_dir=$(echo ${agt_dir%\\*})
##echo "DBG: agt_dir=$agt_dir"
BSERVCMD="${agt_dir}\\${BSERV_CMDNAM}"
# get agent's user
agt_usr=$(./boomCMD ${node} "whoami"); ret=$?
if [ ${ret} -ne 0 ]
then
  ShowT WARNING "(${PID}) [${user}] - node ${node_short} not available!"
  exit 9
fi

ShowT INFO "(${PID}) [${user}] ${NAME} started for ${node_short} >>${agt_dir}<< (${agt_usr})"

# write cmd-file bserv.cmd
echo "TIMEOUT /T 1" >${BSERV_CMDFIL}
echo "\"${agt_dir}\\${BOOMASERV}\" -u" >>${BSERV_CMDFIL}
echo "TIMEOUT /T 3" >>${BSERV_CMDFIL}
echo "copy /Y \"${agt_dir}\\510${BOOMASERV}\" \"${agt_dir}\\${BOOMASERV}\"" >>${BSERV_CMDFIL}
echo "TIMEOUT /T 1" >>${BSERV_CMDFIL}
echo "\"${agt_dir}\\${BOOMASERV}\" -i" >>${BSERV_CMDFIL}
echo "TIMEOUT /T 3" >>${BSERV_CMDFIL}
echo "\"${agt_dir}\\${BOOMASERV}\" -s" >>${BSERV_CMDFIL}

filetime=$(date -r "${BSERV_CMDFIL}" +"%s")
now=$(date +"%s")
timediff=$((now-filetime))
filelen=$(wc -l ${BSERV_CMDFIL} | awk '{print $1}')
if [ ${timediff} -ge ${WRITE_INTERV} ] || [ ${filelen} -ne ${BSERV_CMDFIL_LEN} ]
then
  ShowT ERROR "(${PID}) [${user}] ${node_short} - cmd-file >>${BSERV_CMDFIL}<< not up-to-date (modification time > ${timediff}s/file length = ${filelen})!"
  exit 8
fi

# check service entry
ShowT INFO "(${PID}) [${user}] ${node_short} - print Service Path: "
res=$(./boomCMD ${node} "wmic service get name,displayname,pathname,startmode  | findstr /i \"boomagent\" ") ; ret=$?
if [ ${ret} -ne 0 ]
then
  ShowT ERROR "(${PID}) [${user}] ${node_short} - get service info failed!"
  exit 6
else
  echo "${res}"
fi


# upload 1: overwrite boom_agt.cmd
./boomUPLD ${node} ${SRVDEPLOY_AGTDIR}/${win}/${BOOMACMD} "${BOOMACMD}" 1>/dev/null; ret=$?
if [ ${ret} -ne 0 ]
then
  ShowT ERROR "(${PID}) [${user}] ${node_short} - upload ${BOOMACMD} (${agt_dir}) failed!"
  exit 2
else
  ShowT INFO "(${PID}) [${user}] ${node_short} - ${BOOMACMD} uploaded (${agt_dir})"
fi

# upload 2: temp. 510...
./boomUPLD ${node} ${SRVDEPLOY_AGTDIR}/${win}/${BOOMASERV} "510${BOOMASERV}" 1>/dev/null; ret=$?
if [ ${ret} -ne 0 ]
then
  ShowT ERROR "(${PID}) [${user}] ${node_short} - upload 510${BOOMASERV} (${agt_dir}) failed!"
  exit 3
else
  ShowT INFO "(${PID}) [${user}] ${node_short} - 510${BOOMASERV} uploaded (${agt_dir})"
fi

# upload 3: cmd-file
./boomUPLD ${node} ${BSERV_CMDFIL} "${BSERV_CMDNAM}" 1>/dev/null; ret=$?
if [ ${ret} -ne 0 ]
then
  ShowT ERROR "(${PID}) [${user}] ${node_short} - upload ${BSERV_CMDNAM} (${agt_dir}) failed!"
  exit 4
else
  ShowT INFO "(${PID}) [${user}] ${node_short} - ${BSERV_CMDNAM} uploaded (${agt_dir})"
fi

# create scheduled task: ONIDLE's cmd-file (run user: system)
./boomCMD ${node} "SCHTASKS /CREATE /SC ONIDLE /TN \"bServ\" /TR \"cmd /c '`echo "${BSERVCMD}"`' \" /RU \"\" /I 1" 1>/dev/null; ret=$?
if [ ${ret} -ne 0 ]
then
  ShowT ERROR "(${PID}) [${user}] ${node_short} - SCHTASKS: create task >>bServ<< failed!"
  exit 7
fi

# run task: start now
./boomCMD ${node} "SCHTASKS /RUN /TN \"bServ\"" 1>/dev/null; ret=$?
if [ ${ret} -ne 0 ]
then
  ShowT ERROR "(${PID}) [${user}] ${node_short} - SCHTASKS: run task >>bServ<< failed!"
  exit 7
else
  ShowT INFO "(${PID}) [${user}] ${node_short} - create and start scheduled task >>bServ<< NOW: aborts connection ... (however onsite \"background\" execution: uninstall, overwrite, install, start service ...)"
fi

[[ ${ss_flag} -eq 1 ]] && {
  ShowStarCC
}

# connection aborted: try for reconnect provided by Agent
for ((cnt=1;$cnt<${MAXWAIT_CONNECT};cnt++))
do
  ./boomCMD ${node} "cd" 1 >/dev/null; ret=$?
##echo "DBG: ret=$ret"
  if [ ${ret} -ne 0 ]
  then
# ONLINE
    break
  fi
  sleep 1
done

[[ ${ss_flag} -eq 1 ]] && {
  kill -s SIGINT ${ShowStarCC_PID}; echo -en "\r \r"
}

if [ ${cnt} -ge ${MAXWAIT_CONNECT} ]
then
  ShowT WARNING "(${PID}) [${user}] ${node_short} - Agent still online last ${MAXWAIT_CONNECT} sec. while service restart should happened ..."
else
  ShowT INFO "(${PID}) [${user}] ${node_short} - Agent connection broken (within last ${cnt} sec.) - wait for re-connect (max. $((MAXWAIT_CONNECT*WAITS))s) ..."
fi

[[ ${ss_flag} -eq 1 ]] && {
  ShowStarC
}

for ((cnt=1;$cnt<${MAXWAIT_CONNECT};cnt++))
do
  ./boomCMD ${node} "cd" 1 >/dev/null; ret=$?
##echo "DBG: ret=$ret"
  if [ ${ret} -eq 0 ]
  then
# ONLINE
    break  
  fi
  sleep ${WAITS} 
done

[[ ${ss_flag} -eq 1 ]] && {
  kill -s SIGINT ${ShowStarC_PID}; echo -en "\r \r"
}

if [ ${cnt} -ge ${MAXWAIT_CONNECT} ]
then
  ShowT ERROR "(${PID}) [${user}] ${node_short} - timeout: Agent offline!"
  ShowT INFO "(${PID}) [${user}] ${node_short} -  check >>${BOOMASERV}<< and start boom Agent service (BOOMAgent) locally"
  exit 5
else
  ((cnt=cnt+1))
  ShowT INFO "(${PID}) [${user}] ${node_short} - boom Agent service restarted after $((cnt*WAITS))s (Agent online)"
fi

# delete scheduled task
./boomCMD ${node} "SCHTASKS /DELETE /TN \"bServ\" /F" 1>/dev/null; ret=$?
if [ ${ret} -ne 0 ]
then
  ShowT ERROR "(${PID}) [${user}] ${node_short} - SCHTASKS: delete task >>bServ<< failed!"
else
  ShowT INFO "(${PID}) [${user}] ${node_short} - delete scheduled task >>bServ<<"
fi

# check service entry
sleep ${WAITS} 
ShowT INFO "(${PID}) [${user}] ${node_short} - print Service Path (${BOOMASERV}): "
res=$(./boomCMD ${node} "wmic service get name,displayname,pathname,startmode  | findstr /i \"boomagent\" ") ; ret=$?
if [ ${ret} -ne 0 ]
then
  ShowT ERROR "(${PID}) [${user}] ${node_short} - get service info (${BOOMASERV}) failed!"
  exit 6
else
  echo "${res}"
  echo "${res}" | grep '".*exe"' >/dev/null
  if [ $? -ne 0 ]
  then
    ShowT WARNING "(${PID}) [${user}] ${node_short} - service path NOT surrounded by double quotes!"
  else
    ShowT INFO "(${PID}) [${user}] ${node_short} - service path surrounded by double quotes"
  fi
fi

cd - >/dev/null
exit 0

