#!/bin/sh
######################################################################
#
#  Name: install_boom_LinuxAgent.sh
#
#  Description: Checks prequisites (Java), creates agent-user (boom:boom),
#               provides Sudoers-File and installs (silent) boom agent on
#               target UNIX servers.
#               passwd -n -1 -x -1 -w -1 -i -1 $cmd_user_name ==> no login;
#               password never expires.
#
#  Author: Eike Krautter, Worldline Germany
#  Version: 2.10
#  Date: 30.9.2021
#
#  Input parameter: [$1 -u (uninstall)]
#
#  Return: 0 = Success
#          1 = USAGE
#          2 = ERROR: only root user can install an agent - installation aborted!
#          3 = ERROR: no java found inside >>PATH<< - installation aborted!
#          4 = ERROR: wrong java version (${JAVA_VER}) - installation aborted!
#          5 = ERROR: Zip-File for agent SW ($cmd_agent_zip) n/a in working directory - installation aborted!
#          6 = WARNING: directory >>${cmd_agent_path}<< does already exist - installation stopped!
#          7 = ERROR: sudoers.d as directory is not present (OS is too old) - installation aborted!
#          8 = ERROR: boom Agent process n/a - installation failed!
#          9 = ERROR: boom Agent is not running after ${MAXWAIT}s - installation failed! 
#
#  Note: environment variable JAVA_HOME_BOOM is optional; but if set path must
#        end with "/bin"
#
#  History:
#
#     date    |name| reason
#  ----------------------------------------------------------------
#  30/09/2021 | ek | (1.00) Initial Release
#  11/02/2022 | ps | (1.01) adjustments
#  31/05/2022 | ps | (2.00) JAVA_PATH: if set via >>export BOOM_JAVA_HOME<<
#                    path checked and passed to "install" which transfers 
#                    JAVA_PATH to newly created boom_agt.cfg 
#  05/07/2022 | ps | (2.01) java_path=${JAVA_HOME_BOOM};
#                    variables: user_id, group_id
#  16/08/2022 | ps | (2.02) path for boom_agt set to cmd_agent_path
#  20/02/2024 | ps | (2.10) boom Agent 5.10.0 supported Java versions: 8, 17, 21
#
# Disclaimer: 
# This information are provided "AS IS" without warranty of any kind, either expressed or implied. 
# The entire risk arising out of the use or performance of the script and documentation remains with you. 
# Furthermore, Worldline or the author shall not be liable for any damages you may sustain by using this information, 
# whether direct, indirect, special, incidental or consequential, including, without limitation, damages for loss of business profits,
# business interruption, loss of business information or other pecuniary loss even if it has been advised of the possibility of such damages.
# Read all the implementation and usage notes thoroughly.
######################################################################

## Customer-specific adaptions: boom Agt Dir, User, Group, UID, GID, JAVA_PATH
cmd_agent_path=/opt/boom/agent
cmd_user_name=boom
cmd_user_group=boom
user_id=""
group_id=""
java_path=${JAVA_HOME_BOOM}

## Constants
cmd_agent_zip=linux64_boom_agent.zip
cmd_path2sudoers=/etc/sudoers.d/imogw_boomagt_sudoers_support
MAXWAIT=10
JAVAPATH_STR="JAVA_PATH"

JAVA_8_VER=8
JAVA_17_VER=17
JAVA_21_VER=21

# FUNCTIONS++++++++++++++++++++++++++++++++++++++++++++++++++
 usage()
 {
 echo "Usage: Run the script without any arguments."
 echo "-u for uninstallation is the only valid argument!"
 echo "The script is going to create a user ($cmd_user_name) with specific rights or otherwise"
 echo "modify the user if it exists. "
 exit 1
 }

 uninstall()
 {
 ShowT INFO "Uninstalling all components of boom agent as well as user and its settings"

# stop boom
  ShowT INFO "stopping the agent"
  $cmd_agent_path/boom_agt -stop

  ShowT INFO "uninstalling the agent"
  $cmd_agent_path/install -u -s
# delete folder
  ShowT INFO "removing agent folder"
  rm -rf $cmd_agent_path
# delete user with home directory
  ShowT INFO "deleting user and its home dir"
  userdel $cmd_user_name -rf
# delete group
  ShowT INFO "deleting group"
  groupdel $cmd_user_group
# delete sudoers file
  ShowT INFO "removing sudoers file"
  rm -f $cmd_path2sudoers
# modify sshd_config
#Todo check if this is necessary
##  echo "Deleting name from sshd_config"
##  sed -i "s/$cmd_user_name//g" /etc/ssh/sshd_config

  ShowT INFO "Uninstall done"
  exit 0
 }

 ShowT()
{
  local timstmp=$(date "+%d.%m.%Y %T")

   echo -e "${timstmp} ${1}: ${2}"
}


# MAIN++++++++++++++++++++++++++++++++++++++++++++++++++++++++

##Check for unnecessary flags
if  [ ! -z "${1}" ] && [ $1 == "-u" ];  then
  ShowT INFO "uninstallation of boom agent started"
  uninstall;
elif [ $# -ne 0 ]; then
   usage;
fi

## Check if the operator is root. If not exit

if [ "$(whoami)" != "root" ]; then
  ShowT ERROR "only root user can install an agent - installation aborted!"
  exit 2
else
  ShowT INFO "user is root"
fi

#Is java installed and correct version

# JAVA_PATH handling
if [ ! -z "${java_path}" ]
then
  cmd_java_path=${java_path}
  if [ -x ${cmd_java_path}/java ]
  then
    ShowT INFO "found java executable inside >>${cmd_java_path}<<"
  else
    ShowT ERROR "no java found inside >>${cmd_java_path}<< - installation aborted!"
    exit 3
  fi
else
 cmd_java_path=`type -p java`

 if type -p java; then
    ShowT INFO "found java executable inside PATH"
    cmd_java_path=`dirname $cmd_java_path`
 else
    ShowT ERROR "no java found inside >>PATH<< - installation aborted!"
    exit 3
 fi
fi

##echo "DBG: cmd_java_path=$cmd_java_path"
JAVA_VER=$(${cmd_java_path}/java -version 2>&1 | sed -n ';s/.* version "\(.*\)\.\(.*\)\..*".*/\1\2/p;')
# 18 or 8 ...
##if [[ "$JAVA_VER" =~ "$JAVA_VER_MAJOR"(\.|$) ]]
if [[ ${JAVA_VER} =~ ${JAVA_8_VER}(\.|$) ]] || [[ ${JAVA_VER} =~ ${JAVA_17_VER} ]] || [[ ${JAVA_VER} =~ ${JAVA_21_VER} ]]
then
  ShowT INFO "Java version ${JAVA_VER} is supported"
else
  ShowT ERROR "Java version ${JAVA_VER} is not supported - installation aborted!"
  exit 4
fi

# Is the agent package in the working directory

if [ -e $cmd_agent_zip ]; then
 ShowT INFO "agent SW package >>${cmd_agent_zip}<< available"
else
  ShowT ERROR "Zip-File for agent SW ($cmd_agent_zip) n/a in working directory - installation aborted!"
  exit 5
fi

#check if the path exists

if [ -d $cmd_agent_path ]; then
  ShowT WARNING "directory >>${cmd_agent_path}<< does already exist - installation stopped!"
  exit 6
fi

#check version of sudo, if too low sudoers.d not present.

if [ -d /etc/sudoers.d/ ]; then
 ShowT INFO "sudoers.d as directory is present"
else
  ShowT ERROR "sudoers.d as directory is not present (OS is too old) - installation aborted!"
  exit 7
fi

ShowT INFO "Prerequisite checks done. Starting Preparation."

##Does the group to run the agent exist? If not create it correctly

if [ "$cmd_user_group:x" == "$(grep -o $cmd_user_group:x /etc/group)" ]; then
  ShowT INFO "Group $cmd_user_group exists - skipping group creation"
else
  ShowT INFO "Group $cmd_user_group does not exist"
  ShowT INFO "creating group >>${cmd_user_group} (${group_id})<<"
  if [ -z "${group_id}" ]
  then
    groupadd $cmd_user_group
  else
    groupadd $cmd_user_group -g $group_id
  fi
fi

##Does the user to run the agent exist? If not create it correctly

if [ "$cmd_user_name:x" == "$(grep -o $cmd_user_name:x /etc/passwd)" ]; then
  ShowT INFO "User $cmd_user_name exists - skipping user creation"

  #Checking /etc/shadow for correct entry of user

  ShowT INFO "Checking /etc/shadow for $cmd_user_name"

  if [ "::::::" == "$(grep $cmd_user_name: /etc/shadow | grep -o "::::::")" ]; then
    ShowT INFO "Settings of $cmd_user_name in /etc/shadow seem to be correct"
  else
    ShowT INFO "Adjusting /etc/shadow for $cmd_user_name (no login; password never expires)"
    passwd -n -1 -x -1 -w -1 -i -1 $cmd_user_name
  fi

else
  ShowT INFO "user $cmd_user_name does not exist"
  ShowT INFO "creating user >>${cmd_user_name} (${user_id})<< (no login; password never expires)"

#adding the user
  if [ -z "${user_id}" ]
  then
    useradd $cmd_user_name -c "boom Agent"
  else
    useradd $cmd_user_name -c "boom Agent" -g $group_id -u $user_id
  fi
  passwd -n -1 -x -1 -w -1 -i -1 $cmd_user_name
fi

#Checking sshd_config and adding an entry to prevent the user from login.
##if [ -z "$(grep DenyUsers /etc/ssh/sshd_config)" ];   then
##  ShowT INFO "Changing sshd_config for $cmd_user_name"
##  echo "DenyUsers $cmd_user_name" >> /etc/ssh/sshd_config
##else
##  ShowT INFO "Changing sshd_config for $cmd_user_name"
  #Note I hope it does not make a problem if a list of DenyUsers is already present in the file.
  #This would lead to a doublicated entry for the user.
##  sed -i "s/DenyUsers/DenyUsers $cmd_user_name/g" /etc/ssh/sshd_config
##fi

##If the user existed, does it have correct sudo rights and pw settings. If none/not found, set them, if sudo config found stop

if [ -f $cmd_path2sudoers ]; then
  ShowT INFO "file $cmd_path2sudoers does exist (assuming correct content) - skipping file creation"
else
  ShowT INFO "providing $cmd_path2sudoers with sudo settings for user $cmd_user_name."
  echo "## Allow the functional user which runs the boom agent to use 3 commands" > $cmd_path2sudoers
  echo "User_Alias IMOGW_BOOMAGT_UA=$cmd_user_name" >> $cmd_path2sudoers
  echo "Cmnd_Alias IMOGW_BOOMAGT_CA=\\" >> $cmd_path2sudoers
  echo "/bin/systemctl * boom_agt*" >> $cmd_path2sudoers
  echo "IMOGW_BOOMAGT_UA ALL = (root) EXEC: NOPASSWD: IMOGW_BOOMAGT_CA " >> $cmd_path2sudoers
  chmod ug-w /etc/sudoers.d/imogw_boomagt_sudoers_support
fi

## create the path to the agent folder and change the owner to the user

ShowT INFO " create >>${cmd_agent_path}<< and change owner to >>${cmd_user_name}:${cmd_user_group}<< (and parent dir)"
  mkdir -p $cmd_agent_path
  unzip $cmd_agent_zip -d $cmd_agent_path
  chown -R $cmd_user_name:$cmd_user_group $cmd_agent_path
workdir=$(pwd)
cd ${cmd_agent_path}
cd ..
chown ${cmd_user_name}:${cmd_user_group} .
cd ${workdir}
  chmod 750 $cmd_agent_path/install

#install silently

ShowT INFO "start silent installation of boom agent"
$cmd_agent_path/install -s -us $cmd_user_name -j $cmd_java_path

##Check if boom is running and process visible

ShowT INFO "waiting for agent running (max. ${MAXWAIT}s)"

cnt=0
until [ ${cnt} -ge ${MAXWAIT} ]
do
  cnt=`expr $cnt + 1`
  sleep 1
  ShowT INFO "(${cnt}/${MAXWAIT}) check boom agent status"
  if [ "Boom Agent is running" == "$(${cmd_agent_path}/boom_agt -status)" ] 
  then
    ShowT INFO "boom Agent is running after ${cnt}s"
    sleep 1
    if [ "DBOOMAGENTPROC" == "$(ps -ef | grep -v grep | grep -o DBOOMAGENTPROC)" ]
    then
      ShowT INFO "boom Agent process ready - installation done"
      exit 0
    else
      ShowT ERROR "boom Agent process n/a - installation failed!"
      exit 8
    fi
  fi
done

ShowT ERROR "boom Agent is not running after ${MAXWAIT}s - installation failed!"
exit 9


