#!/bin/sh
######################################################################
#
#  Name: update_boom_LinuxAgent.sh
#
#  Description: Upgrades a boom agent to the version of a zip file provided under working directory.
#
#  Author: Eike Krautter, Worldline Germany
#  Version: 2.10
#  Date: 22.02.2022
#
#  Input parameter: No input argument is accepted
#
#  Return: 0 = Success
#          1 = USAGE
#          2 = ERROR: only root user can install an agent - upgrade aborted!
#          3 = ERROR: no java found inside >>PATH<< - upgrade aborted!
#          4 = ERROR: wrong java version (${JAVA_VER}) - upgrade aborted!
#          5 = ERROR: Zip-File for agent SW ($cmd_agent_zip) n/a in working directory - upgrade aborted!
#          6 = WARNING: directory >>${cmd_agent_path}<< does not yet exist - upgrade stopped!
#          7 = ERROR: sudoers.d as directory is not present (OS is too old) - upgrade aborted!
#          8 = ERROR: boom Agent process n/a - upgrade failed!
#          9 = ERROR: boom Agent is not running after ${MAXWAIT}s - upgrade failed!
#          10 = ERROR: Group $cmd_user_group does not exist
#          11 = ERROR: user $cmd_user_name does not exist
#
#  Note: environment variable JAVA_HOME_BOOM is optional; but if set path must
#        end with "/bin"
#
#  History:
#
#     date    |name| reason
#  ----------------------------------------------------------------
#  22/02/2022 | ek | (1.00) Initial Release
#  31/05/2022 | ps | (2.00) JAVA_PATH: if set via >>export BOOM_JAVA_HOME<<
#                    path checked and and written to boom_agt.cfg if different
#  05/07/2022 | ps | (2.01) java_path=${JAVA_HOME_BOOM}
#  20/02/2024 | ps | (2.10) boom Agent 5.10.0 supported Java versions: 8, 17, 21
#
# Disclaimer: 
# This information are provided "AS IS" without warranty of any kind, either expressed or implied. 
# The entire risk arising out of the use or performance of the script and documentation remains with you. 
# Furthermore, Worldline or the author shall not be liable for any damages you may sustain by using this information, 
# whether direct, indirect, special, incidental or consequential, including, without limitation, damages for loss of business profits,
# business interruption, loss of business information or other pecuniary loss even if it has been advised of the possibility of such damages.
# Read all the implementation and usage notes thoroughly.
######################################################################

## Customer-specific adaptions: boom Agt Dir, User, Group, JAVA_PATH
cmd_agent_path=/opt/boom/agent
cmd_user_name=boom
cmd_user_group=boom
tmp_agent_dir=tmpDir
java_path=${JAVA_HOME_BOOM}

## Constants
cmd_agent_zip=linux64_boom_agent.zip
MAXWAIT=10
JAVAPATH_STR="JAVA_PATH"

JAVA_8_VER=8
JAVA_17_VER=17
JAVA_21_VER=21

CFG_FILE="boom_agt.cfg"


# FUNCTIONS++++++++++++++++++++++++++++++++++++++++++++++++++
 usage()
 {
 echo "Usage: Run the script without any arguments."
 echo "Usage: the zip-file must be available in the working directory"
 exit 1
 }

 ShowT()
{
  local timstmp=$(date "+%d.%m.%Y %T")

   echo -e "${timstmp} ${1}: ${2}"
}

# MAIN++++++++++++++++++++++++++++++++++++++++++++++++++++++++

##Check for unnecessary flags
if  [ -n "${0}" ];  then
  ShowT INFO "upgrade of boom agent started"
elif [ $# -ne 0 ]; then
   usage;
fi

## Check if the operator is root. If not exit

if [ "$(whoami)" != "root" ]; then
  ShowT ERROR "only root user can upgrade an agent - upgrade aborted!"
  exit 2
else
  ShowT INFO "user is root"
fi

#Is java installed and correct version
# JAVA_PATH handling
JAVA_PATH_CONF=$(grep ${JAVAPATH_STR} "${cmd_agent_path}/${CFG_FILE}")
if [ $? -ne 0 ]
then
# no entry in cfg
  ShowT INFO "${JAVAPATH_STR} is not set in ${cmd_agent_path}/${CFG_FILE}"
  if [ ! -z "${java_path}" ]
  then
# Append JAVA_PATH in cfg
    ShowT INFO "appending >>${JAVAPATH_STR}=${java_path}<< to ${cmd_agent_path}/${CFG_FILE}"
    printf "${JAVAPATH_STR}=${java_path}\n" >>${cmd_agent_path}/${CFG_FILE}
  fi
else
# also remove ""
  JAVA_PATH_CONF=$(echo ${JAVA_PATH_CONF} |awk -F"=" '{print $2}' |sed 's/\"//g')
  ShowT INFO "${JAVAPATH_STR} (${java_path}) is already set to >>${JAVA_PATH_CONF}<< in ${cmd_agent_path}/${CFG_FILE}"
  if [ ! -z "${java_path}" ]
  then
#echo "DBG: JAVA_PATH_CONF=$JAVA_PATH_CONF java_path=$java_path"
    if [ "${JAVA_PATH_CONF}" != "${java_path}" ]
    then
# Replace JAVA_PATH in cfg
      ShowT INFO "replacing >>${JAVAPATH_STR}=${java_path}<< in ${cmd_agent_path}/${CFG_FILE}"
      cat ${cmd_agent_path}/${CFG_FILE} | grep -v ${JAVAPATH_STR} >${cmd_agent_path}/${CFG_FILE}.tmp
      printf "${JAVAPATH_STR}=${java_path}\n" >>${cmd_agent_path}/${CFG_FILE}.tmp
      mv -f ${cmd_agent_path}/${CFG_FILE}.tmp ${cmd_agent_path}/${CFG_FILE}
    fi
  fi
fi

if [ ! -z "${java_path}" ]
then
  cmd_java_path=${java_path}
  if [ -x ${cmd_java_path}/java ]
  then
    ShowT INFO "found java executable inside >>${cmd_java_path}<<"
  else
    ShowT ERROR "no java found inside >>${cmd_java_path}<< - upgrade aborted!"
    exit 3
  fi
else
 cmd_java_path=`type -p java`

 if type -p java; then
    ShowT INFO "found java executable inside PATH"
    cmd_java_path=`dirname $cmd_java_path`
 else
    ShowT ERROR "no java found inside >>PATH<< - upgrade aborted!"
    exit 3
 fi
fi


##echo "DBG: cmd_java_path=$cmd_java_path"
JAVA_VER=$(${cmd_java_path}/java -version 2>&1 | sed -n ';s/.* version "\(.*\)\.\(.*\)\..*".*/\1\2/p;')
# 18 or 8 ...
##if [[ "$JAVA_VER" =~ "$JAVA_VER_MAJOR"(\.|$) ]]
if [[ ${JAVA_VER} =~ ${JAVA_8_VER}(\.|$) ]] || [[ ${JAVA_VER} =~ ${JAVA_17_VER} ]] || [[ ${JAVA_VER} =~ ${JAVA_21_VER} ]]
then
  ShowT INFO "Java version ${JAVA_VER} is supported"
else
  ShowT ERROR "Java version ${JAVA_VER} is not supported - upgrade aborted!"
  exit 4
fi

# Is the agent package in the working directory

if [ -e $cmd_agent_zip ]; then
 ShowT INFO "agent SW package >>${cmd_agent_zip}<< available"
else
  ShowT ERROR "Zip-File for agent SW ($cmd_agent_zip) n/a in working directory - upgrade aborted!"
  exit 5
fi

#check if the path exists

if [ ! -d $cmd_agent_path ]; then
  ShowT WARNING "directory >>${cmd_agent_path}<< does not exist yet - upgrade stopped!"
  exit 6
fi

#check version of sudo, if too low sudoers.d not present.

if [ -d /etc/sudoers.d/ ]; then
 ShowT INFO "sudoers.d as directory is present"
else
  ShowT ERROR "sudoers.d as directory is not present (OS is too old) - upgrade aborted!"
  exit 7
fi

##Does the group to run the agent exist? If not create it correctly

if [ "$cmd_user_group:x" == "$(grep -o $cmd_user_group:x /etc/group)" ]; then
  ShowT INFO "Group $cmd_user_group exists - skipping group creation"
else
  ShowT ERROR "Group $cmd_user_group does not exist"
  exit 10
fi

##Does the user to run the agent exist? If not create it correctly

if [ "$cmd_user_name:x" == "$(grep -o $cmd_user_name:x /etc/passwd)" ]; then
  ShowT INFO "User $cmd_user_name exists - skipping user creation"

  #Checking /etc/shadow for correct entry of user

  ShowT INFO "Checking /etc/shadow for $cmd_user_name"

  if [ "::::::" == "$(grep $cmd_user_name: /etc/shadow | grep -o "::::::")" ]; then
    ShowT INFO "Settings of $cmd_user_name in /etc/shadow seem to be correct"
  else
    ShowT WARNING "Settings in /etc/shadow for $cmd_user_name (no login; password never expires) are not set as expected."
  fi

else
  ShowT ERROR "user $cmd_user_name does not exist"
  exit 11
fi

ShowT INFO "Prerequisite checks done. Starting Preparation."

## unzipping the folder, copy the jar, change ownership remove temp dir.

if [ -d $tmp_agent_dir ]; then
    ShowT INFO "Removing $tmp_agent_dir because it preexisted"
    rm -rf $tmp_agent_dir
fi

ShowT INFO "Unzipping the $cmd_agent_zip to $tmp_agent_dir"
unzip $cmd_agent_zip -d $tmp_agent_dir
ShowT INFO "Stopping the Boom Agent"
$cmd_agent_path/boom_agt -stop
ShowT INFO "Copy the boom_agt.jar to $cmd_agent_path"
cp $tmp_agent_dir/boom_agent.jar $cmd_agent_path
ShowT INFO "changing the ownership of boom_agent.jar under $cmd_agent_path"
chown -R $cmd_user_name:$cmd_user_group $cmd_agent_path/boom_agent.jar
ShowT INFO "Removing $tmp_agent_dir"
rm -rf $tmp_agent_dir
ShowT INFO "Starting the Boom Agent"
$cmd_agent_path/boom_agt -start

##Check if boom is running and process visible

ShowT INFO "waiting for agent running (max. ${MAXWAIT}s)"

cnt=0
until [ ${cnt} -ge ${MAXWAIT} ]
do
  cnt=`expr $cnt + 1`
  sleep 1
  ShowT INFO "(${cnt}/${MAXWAIT}) check boom agent status"
  if [ "Boom Agent is running" == "$($cmd_agent_path/boom_agt -status)" ]
  then
    ShowT INFO "boom Agent is running after ${cnt}s"
    sleep 1
    if [ "DBOOMAGENTPROC" == "$(ps -ef | grep -v grep | grep -o DBOOMAGENTPROC)" ]
    then
      ShowT INFO "boom Agent process ready - upgrade done"
      exit 0
    else
      ShowT ERROR "boom Agent process n/a - upgrade failed!"
      exit 8
    fi
  fi
done

ShowT ERROR "boom Agent is not running after ${MAXWAIT}s - upgrade failed!"
exit 9
